from __future__ import annotations
import os
import shutil
from fastapi import FastAPI, Depends, HTTPException
from fastapi.middleware.cors import CORSMiddleware
from sqlalchemy.orm import Session
from geoalchemy2.shape import from_shape
from shapely.geometry import Point, shape
from .db import engine, Base, get_db
from . import models, schemas
from .auth import verify_pw, create_token
from .deps import get_current_user
from .bootstrap import ensure_demo_data
from .tasks import enqueue_process_media
from .storage import ensure_media_root, site_media_dir

app = FastAPI(title="HazardMap Québec API", version="0.1.0")

CORS_ORIGINS = os.getenv("CORS_ORIGINS", "http://localhost:3000").split(",")
app.add_middleware(
    CORSMiddleware,
    allow_origins=[o.strip() for o in CORS_ORIGINS if o.strip()],
    allow_credentials=True,
    allow_methods=["*"],
    allow_headers=["*"],
)

@app.on_event("startup")
def on_startup():
    ensure_media_root()
    Base.metadata.create_all(bind=engine)
    # Ensure PostGIS extension exists (best effort)
    with engine.connect() as conn:
        conn.exec_driver_sql("CREATE EXTENSION IF NOT EXISTS postgis;")
        conn.commit()
    # demo user
    db = next(get_db())
    try:
        ensure_demo_data(db)
    finally:
        db.close()

@app.get("/health")
def health():
    return {"ok": True, "bucket": S3_BUCKET}

@app.post("/auth/login", response_model=schemas.TokenOut)
def login(payload: schemas.LoginIn, db: Session = Depends(get_db)):
    user = db.query(models.User).filter(models.User.email == payload.email).first()
    if not user or not verify_pw(payload.password, user.pw_hash):
        raise HTTPException(status_code=401, detail="Invalid credentials")
    token = create_token(user.id, user.org_id, user.role)
    return {"access_token": token}

@app.get("/me")
def me(user=Depends(get_current_user)):
    return {"id": user.id, "email": user.email, "role": user.role, "org_id": user.org_id}

@app.post("/sites", response_model=schemas.SiteOut)
def create_site(payload: schemas.SiteCreate, user=Depends(get_current_user), db: Session = Depends(get_db)):
    centroid = None
    if payload.centroid_lat is not None and payload.centroid_lng is not None:
        centroid = from_shape(Point(payload.centroid_lng, payload.centroid_lat), srid=4326)
    site = models.Site(org_id=user.org_id, name=payload.name, address=payload.address, centroid=centroid)
    db.add(site); db.commit(); db.refresh(site)
    c = None
    if payload.centroid_lat is not None and payload.centroid_lng is not None:
        c = {"lat": payload.centroid_lat, "lng": payload.centroid_lng}
    return {"id": site.id, "name": site.name, "address": site.address, "centroid": c}

@app.get("/sites", response_model=list[schemas.SiteOut])
def list_sites(user=Depends(get_current_user), db: Session = Depends(get_db)):
    sites = db.query(models.Site).filter(models.Site.org_id == user.org_id).order_by(models.Site.created_at.desc()).all()
    out=[]
    for s in sites:
        centroid=None
        if s.centroid:
            # WKT like "POINT(lng lat)"
            wkt = db.scalar(s.centroid.ST_AsText())
            if wkt and wkt.startswith("POINT"):
                inside = wkt[wkt.find("(")+1:wkt.find(")")]
                lng, lat = [float(x) for x in inside.split()]
                centroid={"lat": lat, "lng": lng}
        out.append({"id": s.id, "name": s.name, "address": s.address, "centroid": centroid})
    return out

@app.post("/sites/{site_id}/zones", response_model=schemas.ZoneOut)
def create_zone(site_id: str, payload: schemas.ZoneCreate, user=Depends(get_current_user), db: Session = Depends(get_db)):
    site = db.query(models.Site).filter(models.Site.id==site_id, models.Site.org_id==user.org_id).first()
    if not site:
        raise HTTPException(404, "Site not found")
    poly = shape(payload.polygon)
    zone = models.Zone(site_id=site.id, name=payload.name, polygon=from_shape(poly, srid=4326))
    db.add(zone); db.commit(); db.refresh(zone)
    return {"id": zone.id, "name": zone.name, "polygon": payload.polygon}


from fastapi import UploadFile, File, Form
from pathlib import Path

@app.post("/media/upload", response_model=schemas.MediaOut)
def upload_media(
    site_id: str = Form(...),
    media_type: str = Form(...),
    file: UploadFile = File(...),
    gps_lat: float | None = Form(None),
    gps_lng: float | None = Form(None),
    heading: float | None = Form(None),
    altitude: float | None = Form(None),
    user=Depends(get_current_user),
    db: Session = Depends(get_db)
):
    site = db.query(models.Site).filter(models.Site.id==site_id, models.Site.org_id==user.org_id).first()
    if not site:
        raise HTTPException(404, "Site not found")

    temp = models.Media()
    media_id = temp.id

    folder = site_media_dir(user.org_id, site_id, media_id)
    Path(folder).mkdir(parents=True, exist_ok=True)
    storage_path = str(Path(folder) / file.filename)

    with open(storage_path, "wb") as out:
        shutil.copyfileobj(file.file, out)

    gps = None
    if gps_lat is not None and gps_lng is not None:
        gps = from_shape(Point(gps_lng, gps_lat), srid=4326)

    m = models.Media(
        id=media_id,
        site_id=site_id,
        uploader_id=user.id,
        media_type=media_type,
        s3_key=storage_path,  # local path
        gps=gps,
        heading=heading,
        altitude=altitude,
        device_meta={"filename": file.filename, "content_type": file.content_type},
        blur_faces_applied=False,
    )
    db.add(m); db.commit()

    enqueue_process_media(media_id)
    return {"id": media_id, "upload_url": "", "s3_key": storage_path}

@app.post("/media/{media_id}/finalize")
def finalize_media(media_id: str, payload: schemas.FinalizeIn, user=Depends(get_current_user), db: Session = Depends(get_db)):
    m = db.query(models.Media).join(models.Site, models.Media.site_id==models.Site.id).filter(
        models.Media.id==media_id, models.Site.org_id==user.org_id
    ).first()
    if not m:
        raise HTTPException(404, "Media not found")
    # enqueue processing (worker will blur faces if enabled)
    enqueue_process_media(media_id)
    return {"queued": True, "media_id": media_id}

@app.get("/sites/{site_id}/hazards", response_model=list[schemas.HazardOut])
def list_hazards(site_id: str, user=Depends(get_current_user), db: Session = Depends(get_db)):
    site = db.query(models.Site).filter(models.Site.id==site_id, models.Site.org_id==user.org_id).first()
    if not site:
        raise HTTPException(404, "Site not found")
    hz = db.query(models.Hazard).filter(models.Hazard.site_id==site_id).order_by(models.Hazard.created_at.desc()).all()
    out=[]
    for h in hz:
        wkt = db.scalar(h.pin.ST_AsText())
        inside = wkt[wkt.find("(")+1:wkt.find(")")]
        lng, lat = [float(x) for x in inside.split()]
        out.append({
            "id": h.id,
            "status": h.status,
            "hazard_type": h.hazard_type,
            "title": h.title,
            "severity": h.severity,
            "pin": {"lat": lat, "lng": lng},
            "primary_rule_tags": h.primary_rule_tags,
        })
    return out

@app.patch("/hazards/{hazard_id}")
def patch_hazard(hazard_id: str, payload: schemas.HazardPatch, user=Depends(get_current_user), db: Session = Depends(get_db)):
    h = db.query(models.Hazard).join(models.Site, models.Hazard.site_id==models.Site.id).filter(
        models.Hazard.id==hazard_id, models.Site.org_id==user.org_id
    ).first()
    if not h:
        raise HTTPException(404, "Hazard not found")
    if payload.status:
        h.status = payload.status
    if payload.assignee_id is not None:
        h.assignee_id = payload.assignee_id
    if payload.title:
        h.title = payload.title
    if payload.description is not None:
        h.description = payload.description
    if payload.pin_lat is not None and payload.pin_lng is not None:
        h.pin = from_shape(Point(payload.pin_lng, payload.pin_lat), srid=4326)
    db.add(h); db.commit()
    return {"ok": True}

@app.get("/media/{media_id}/download")
def media_download(media_id: str, user=Depends(get_current_user), db: Session = Depends(get_db)):
    m = db.query(models.Media).join(models.Site, models.Media.site_id==models.Site.id).filter(
        models.Media.id==media_id, models.Site.org_id==user.org_id
    ).first()
    if not m:
        raise HTTPException(404, "Media not found")
    path = m.s3_key
    if not path or not os.path.exists(path):
        raise HTTPException(404, "File missing")
    def iterfile():
        with open(path, 'rb') as f:
            while True:
                chunk = f.read(1024*1024)
                if not chunk:
                    break
                yield chunk
    fname = os.path.basename(path)
    return StreamingResponse(iterfile(), media_type='application/octet-stream', headers={'Content-Disposition': f'attachment; filename="{fname}"'})
